"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const db_1 = require("@b/db");
const utils_1 = require("../utils");
const constants_1 = require("@b/utils/constants");
const query_1 = require("@b/utils/query");
const errors_1 = require("@b/utils/schema/errors");
exports.metadata = {
    summary: "List all AI Market Maker markets",
    operationId: "listAiMarketMakerMarkets",
    tags: ["Admin", "AI Market Maker", "Market"],
    description: "Retrieves a paginated list of all AI Market Maker markets with their associated pool data, ecosystem market details, and current configuration. Supports filtering, sorting, and searching across market parameters.",
    parameters: constants_1.crudParameters,
    responses: {
        200: {
            description: "List of AI Market Maker markets retrieved successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            items: {
                                type: "array",
                                items: {
                                    type: "object",
                                    properties: {
                                        ...utils_1.aiMarketMakerSchema,
                                        pool: {
                                            type: "object",
                                            description: "Market maker pool balances and P&L",
                                        },
                                        market: {
                                            type: "object",
                                            description: "Associated ecosystem market details",
                                        },
                                    },
                                },
                            },
                            pagination: constants_1.paginationSchema,
                        },
                    },
                },
            },
        },
        401: errors_1.unauthorizedResponse,
        404: (0, errors_1.notFoundResponse)("AI Market Maker Markets"),
        500: errors_1.serverErrorResponse,
    },
    requiresAuth: true,
    logModule: "ADMIN_AI",
    logTitle: "Get Market Maker Markets",
    permission: "view.ai.market-maker.market",
};
exports.default = async (data) => {
    const { query, ctx } = data;
    ctx === null || ctx === void 0 ? void 0 : ctx.step("Get Market Maker Markets");
    ctx === null || ctx === void 0 ? void 0 : ctx.success("Get Market Maker Markets retrieved successfully");
    return (0, query_1.getFiltered)({
        model: db_1.models.aiMarketMaker,
        query,
        sortField: query.sortField || "createdAt",
        paranoid: false,
        includeModels: [
            {
                model: db_1.models.aiMarketMakerPool,
                as: "pool",
            },
            {
                model: db_1.models.ecosystemMarket,
                as: "market",
            },
        ],
    });
};
